/*
* linux-rtos-adaptor.c- Sigmastar
*
* Copyright (c) [2019~2020] SigmaStar Technology.
*
*
* This software is licensed under the terms of the GNU General Public
* License version 2, as published by the Free Software Foundation, and
* may be copied, distributed, and modified under those terms.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License version 2 for more details.
*
*/
#include <linux/module.h>
#include <linux/device.h>
#include <linux/major.h>
#include <linux/fs.h>
#include <linux/arm-smccc.h>
#include <linux/slab.h>
#include <linux/kernel.h>
#include <linux/uaccess.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/mman.h>
#include <linux/semaphore.h>
#include <linux/atomic.h>
#include <asm/pgtable.h>
#include <asm/io.h>
#include <linux/seq_file.h>
#include <linux/proc_fs.h>
#include <linux/poll.h>
#include "drv_dualos.h"
#include "cam_inter_os.h"
#include "ms_platform.h"
#include <mstar_chip.h>


#define __MI_DEVICE_PROC 0xfffffffful
#define __MI_DEVICE_PROC_IO 0xfffffffeul
#define __MI_DEVICE_PROC_READLOG 0xfffffffdul
#define __MI_DEVICE_CONNECT 0
#define __MI_DEVICE_DISCONNECT 1
#define __MI_DEVICE_QUERY 2
#define __MI_DEVICE_POLL_CREATE 3
#define __MI_DEVICE_POLL_RELEASE 4
#define __MI_DEVICE_POLL_STATE 5
#define INTEROS_SC_L2R_MI_MMA_BASE  (0xffffff87)
#define INTEROS_SC_L2R_MI_MMA_SIZE  (0xffffff88)
#define INTEROS_SC_R2L_MI_NOTIFY    (0xffffff89)
const unsigned int INTEROS_SC_L2R_MI_CALL_START =  0xff000000;
const unsigned int INTEROS_SC_L2R_MI_CALL_END   =  0xff000040;

const int ALKAID_RTKTRACE = 0;
//#define MMA_BASE (0x20000000ul+E_MMAP_ID_RTK_mma_heap_ADR)
//#define MMA_SIZE (0x0ul+E_MMAP_ID_RTK_mma_heap_LEN)
#define CTX_NUM 2
#define CTX_BASE 8

#define FIVE_SEC (HZ*5)

typedef enum
{
    E_MI_MODULE_ID_IVE    = 0,
    E_MI_MODULE_ID_VDF      = 1,
    E_MI_MODULE_ID_VENC     = 2,
    E_MI_MODULE_ID_RGN     = 3,
    E_MI_MODULE_ID_AI    = 4,
    E_MI_MODULE_ID_AO    = 5,
    E_MI_MODULE_ID_VIF   = 6,
    E_MI_MODULE_ID_VPE    = 7,
    E_MI_MODULE_ID_VDEC    = 8,
    E_MI_MODULE_ID_SYS     = 9,
    E_MI_MODULE_ID_FB   = 10,
    E_MI_MODULE_ID_HDMI  = 11,
    E_MI_MODULE_ID_DIVP  = 12,
    E_MI_MODULE_ID_GFX   = 13,
    E_MI_MODULE_ID_VDISP   = 14,
    E_MI_MODULE_ID_DISP     = 15,
    E_MI_MODULE_ID_OS     = 16,
    E_MI_MODULE_ID_IAE = 17,
    E_MI_MODULE_ID_MD = 18,
    E_MI_MODULE_ID_OD = 19,
    E_MI_MODULE_ID_SHADOW = 20,
    E_MI_MODULE_ID_WARP = 21,
    E_MI_MODULE_ID_UAC = 22,
    E_MI_MODULE_ID_LDC = 23,
    E_MI_MODULE_ID_SD = 24,
    E_MI_MODULE_ID_PANEL = 25,
    E_MI_MODULE_ID_CIPHER = 26,
    E_MI_MODULE_ID_SNR = 27,
    E_MI_MODULE_ID_WLAN =28,
    E_MI_MODULE_ID_IPU = 29,
    E_MI_MODULE_ID_MIPITX = 30,
    E_MI_MODULE_ID_GYRO = 31,
    E_MI_MODULE_ID_JPD = 32,
    E_MI_MODULE_ID_PSPI = 33,
    //E_MI_MODULE_ID_SED  = 29,
    E_MI_MODULE_ID_MAX,
} MI_ModuleId_e;

#define MMA_MAP_HASH_BITS 8
static CAM_OS_DEFINE_HASHTABLE(g_mma_map_addr_hash,MMA_MAP_HASH_BITS);
CamOsTsem_t g_mma_map_hash_semphore;

typedef struct mma_map_record_s
{
    struct CamOsHListNode_t   hentry;
    s32 pid;
    void *mma_cache_vaddr;
    void *mma_nocache_vaddr;
} mma_map_record_t;

typedef enum
{
    E_MI_SYS_CMD_INIT,
    E_MI_SYS_CMD_EXIT,
    E_MI_SYS_CMD_BIND_CHN_PORT,
    E_MI_SYS_CMD_BIND_CHN_PORT2,
    E_MI_SYS_CMD_UNBIND_CHN_PORT,
    E_MI_SYS_CMD_GET_BIND_BY_DEST,
    E_MI_SYS_CMD_GET_VERSION,
    E_MI_SYS_CMD_GET_CUR_PTS,
    E_MI_SYS_CMD_INIT_PTS_BASE,
    E_MI_SYS_CMD_SYNC_PTS,
    E_MI_SYS_CMD_MMAP,
    E_MI_SYS_CMD_MUNMAP,
    E_MI_SYS_CMD_VA2PA,
    E_MI_SYS_CMD_SET_REG,
    E_MI_SYS_CMD_GET_REG,
    E_MI_SYS_CMD_SET_CHN_MMA_CONF,
    E_MI_SYS_CMD_GET_CHN_MMA_CONF,
    E_MI_SYS_CMD_CHN_INPUT_PORT_GET_BUF,
    E_MI_SYS_CMD_CHN_INPUT_PORT_PUT_BUF,
    E_MI_SYS_CMD_CHN_OUTPUT_PORT_GET_BUF,
    E_MI_SYS_CMD_CHN_OUTPUT_PORT_PUT_BUF,
    E_MI_SYS_CMD_SET_CHN_OUTPUT_PORT_DEPTH,
    E_MI_SYS_CMD_GET_CHN_OUTPUT_PORT_DEPTH,
    E_MI_SYS_CMD_GET_POLL_PRIVATE_DATA,
    E_MI_SYS_CMD_PUT_POLL_PRIVATE_DATA,
    E_MI_SYS_CMD_GET_FD_FROM_CHNPORT,
    E_MI_SYS_CMD_SET_FD_TO_CHNPORT,
    E_MI_SYS_CMD_CHN_PORT_INJECT_BUF,
    E_MI_SYS_MMA_ALLOC,
    E_MI_SYS_MMA_FREE,
    E_MI_SYS_FLUSH_INV_CACHE,
    E_MI_SYS_CONF_DEV_PUB_POOLS,
    E_MI_SYS_REL_DEV_PUB_POOLS,
    E_MI_SYS_CONF_GLO_PUB_POOLS,
    E_MI_SYS_REL_GLO_PUB_POOLS,
    E_MI_SYS_CMD_VDEC_USE_VBPOOL,
    E_MI_SYS_CMD_READ_UUID,
    E_MI_SYS_CONFIG_PRIVATE_MMA_HEAP,
    E_MI_SYS_MEMSET_PA,
    E_MI_SYS_MEMCPY_PA,
    E_MI_SYS_BUF_FILL_PA,
    E_MI_SYS_BUF_BLIT_PA,
    E_MI_SYS_CMD_PRIV_DEV_CHN_HEAP_ALLOC,
    E_MI_SYS_CMD_PRIV_DEV_CHN_HEAP_FREE,
    E_MI_SYS_CMD_CHN_OUTPUTPORT_LOWLATENCY,
    E_MI_SYS_CMD_CHN_DUP_BUF,
    E_MI_SYS_CMD_SET_GLOBAL,
    E_MI_SYS_CMD_MAX,
} MI_SYS_Cmd_e;

//-------------------------------------------------------------------------------------------------
//  System Data Type
//-------------------------------------------------------------------------------------------------
/// data type unsigned char, data length 1 byte
typedef unsigned char                            MI_U8;         // 1 byte
/// data type unsigned short, data length 2 byte
typedef unsigned short                           MI_U16;        // 2 bytes
/// data type unsigned int, data length 4 byte
typedef unsigned int                             MI_U32;        // 4 bytes
/// data type unsigned int, data length 8 byte
typedef unsigned long long                       MI_U64;        // 8 bytes
/// data type signed char, data length 1 byte
typedef signed char                              MI_S8;         // 1 byte
/// data type signed short, data length 2 byte
typedef signed short                             MI_S16;        // 2 bytes
/// data type signed int, data length 4 byte
typedef signed int                               MI_S32;        // 4 bytes
/// data type signed int, data length 8 byte
typedef signed long long                         MI_S64;        // 8 bytes
/// data type float, data length 4 byte
typedef float                                    MI_FLOAT;      // 4 bytes
/// data type 64bit physical address
typedef unsigned long long                       MI_PHY;        // 8 bytes
/// data type pointer content
typedef unsigned long                            MI_VIRT;       // 4 bytes when 32bit toolchain, 8 bytes when 64bit toolchain.

typedef unsigned char                            MI_BOOL;

typedef MI_S32 MI_SYS_BUF_HANDLE;

#define ALIGN_UP(x, align) (((x) + ((align) - 1)) & ~((align) - 1))

#define MI_SYS_MAX_SUB_PLANE_CNT (4)

typedef enum
{
    E_MI_SYS_DATA_PRECISION_8BPP,
    E_MI_SYS_DATA_PRECISION_10BPP,
    E_MI_SYS_DATA_PRECISION_12BPP,
    E_MI_SYS_DATA_PRECISION_14BPP,
    E_MI_SYS_DATA_PRECISION_16BPP,
    E_MI_SYS_DATA_PRECISION_MAX,
} MI_SYS_DataPrecision_e;

typedef enum
{
    E_MI_SYS_PIXEL_BAYERID_RG,
    E_MI_SYS_PIXEL_BAYERID_GR,
    E_MI_SYS_PIXEL_BAYERID_BG,
    E_MI_SYS_PIXEL_BAYERID_GB,
    E_MI_SYS_PIXEL_RGBIR_R0,
    E_MI_SYS_PIXEL_RGBIR_G0,
    E_MI_SYS_PIXEL_RGBIR_B0,
    E_MI_SYS_PIXEL_RGBIR_G1,
    E_MI_SYS_PIXEL_RGBIR_G2,
    E_MI_SYS_PIXEL_RGBIR_I0,
    E_MI_SYS_PIXEL_RGBIR_G3,
    E_MI_SYS_PIXEL_RGBIR_I1,
    E_MI_SYS_PIXEL_BAYERID_MAX,
}MI_SYS_BayerId_e;

typedef enum
{
    E_MI_SYS_PIXEL_FRAME_YUV422_YUYV = 0,
    E_MI_SYS_PIXEL_FRAME_ARGB8888,
    E_MI_SYS_PIXEL_FRAME_ABGR8888,
    E_MI_SYS_PIXEL_FRAME_BGRA8888,

    E_MI_SYS_PIXEL_FRAME_RGB565,
    E_MI_SYS_PIXEL_FRAME_ARGB1555,
    E_MI_SYS_PIXEL_FRAME_ARGB4444,
    E_MI_SYS_PIXEL_FRAME_I2,
    E_MI_SYS_PIXEL_FRAME_I4,
    E_MI_SYS_PIXEL_FRAME_I8,

    E_MI_SYS_PIXEL_FRAME_YUV_SEMIPLANAR_422,
    E_MI_SYS_PIXEL_FRAME_YUV_SEMIPLANAR_420,
    E_MI_SYS_PIXEL_FRAME_YUV_SEMIPLANAR_420_NV21,
    E_MI_SYS_PIXEL_FRAME_YUV_TILE_420,
    E_MI_SYS_PIXEL_FRAME_YUV422_UYVY,
    E_MI_SYS_PIXEL_FRAME_YUV422_YVYU,
    E_MI_SYS_PIXEL_FRAME_YUV422_VYUY,

    E_MI_SYS_PIXEL_FRAME_YUV422_PLANAR,
    E_MI_SYS_PIXEL_FRAME_YUV420_PLANAR,

    E_MI_SYS_PIXEL_FRAME_FBC_420,

    E_MI_SYS_PIXEL_FRAME_RGB_BAYER_BASE,
    E_MI_SYS_PIXEL_FRAME_RGB_BAYER_NUM = E_MI_SYS_PIXEL_FRAME_RGB_BAYER_BASE + E_MI_SYS_DATA_PRECISION_MAX*E_MI_SYS_PIXEL_BAYERID_MAX-1,

    E_MI_SYS_PIXEL_FRAME_RGB888,
    E_MI_SYS_PIXEL_FRAME_BGR888,
    E_MI_SYS_PIXEL_FRAME_GRAY8,
    E_MI_SYS_PIXEL_FRAME_FORMAT_MAX,
} MI_SYS_PixelFormat_e;

#define RGB_BAYER_PIXEL(BitMode, PixelID) (E_MI_SYS_PIXEL_FRAME_RGB_BAYER_BASE+ BitMode*E_MI_SYS_PIXEL_BAYERID_MAX+ PixelID)

typedef enum
{
    E_MI_SYS_COMPRESS_MODE_NONE,//no compress
    E_MI_SYS_COMPRESS_MODE_SEG,//compress unit is 256 bytes as a segment
    E_MI_SYS_COMPRESS_MODE_LINE,//compress unit is the whole line
    E_MI_SYS_COMPRESS_MODE_FRAME,//compress unit is the whole frame
    E_MI_SYS_COMPRESS_MODE_TO_8BIT,
    E_MI_SYS_COMPRESS_MODE_BUTT, //number
}MI_SYS_CompressMode_e;

typedef enum
{
    E_MI_SYS_FRAME_SCAN_MODE_PROGRESSIVE = 0x0,  // progessive.
    E_MI_SYS_FRAME_SCAN_MODE_INTERLACE   = 0x1,  // interlace.
    E_MI_SYS_FRAME_SCAN_MODE_MAX,
} MI_SYS_FrameScanMode_e;

typedef enum
{
    E_MI_SYS_FRAME_TILE_MODE_NONE = 0,
    E_MI_SYS_FRAME_TILE_MODE_16x16,      // tile mode 16x16
    E_MI_SYS_FRAME_TILE_MODE_16x32,      // tile mode 16x32
    E_MI_SYS_FRAME_TILE_MODE_32x16,      // tile mode 32x16
    E_MI_SYS_FRAME_TILE_MODE_32x32,      // tile mode 32x32
    E_MI_SYS_FRAME_TILE_MODE_MAX
} MI_SYS_FrameTileMode_e;

typedef enum
{
    E_MI_SYS_FIELDTYPE_NONE,        //< no field.
    E_MI_SYS_FIELDTYPE_TOP,           //< Top field only.
    E_MI_SYS_FIELDTYPE_BOTTOM,    //< Bottom field only.
    E_MI_SYS_FIELDTYPE_BOTH,        //< Both fields.
    E_MI_SYS_FIELDTYPE_NUM
} MI_SYS_FieldType_e;

typedef enum
{
    E_MI_SYS_BUFDATA_RAW = 0,
    E_MI_SYS_BUFDATA_FRAME,
    E_MI_SYS_BUFDATA_META,
    E_MI_SYS_BUFDATA_MULTIPLANE,
} MI_SYS_BufDataType_e;

typedef enum
{
  REALTIME_FRAME_DATA,
  RINGBUF_FRAME_DATA,
  NORMAL_FRAME_DATA,
}MI_SYS_FrameData_PhySignalType;

typedef enum
{
    E_MI_SYS_FRAME_ISP_INFO_TYPE_NONE,
    E_MI_SYS_FRAME_ISP_INFO_TYPE_GLOBAL_GRADIENT
}MI_SYS_FrameIspInfoType_e;

typedef struct MI_SYS_ChnPort_s
{
    MI_ModuleId_e eModId;
    MI_U32  u32DevId;
    MI_U32 u32ChnId;
    MI_U32 u32PortId;

} MI_SYS_ChnPort_t;

typedef struct MI_SYS_WindowRect_s
{
    MI_U16 u16X;
    MI_U16 u16Y;
    MI_U16 u16Width;
    MI_U16 u16Height;
}MI_SYS_WindowRect_t;

typedef struct MI_SYS_WindowSize_s
{
    MI_U16 u16Width;
    MI_U16 u16Height;
}MI_SYS_WindowSize_t;

typedef struct MI_SYS_RawData_s
{
    void*   pVirAddr;
    MI_PHY  phyAddr;//notice that this is miu bus addr,not cpu bus addr.
    MI_U32  u32BufSize;

    MI_U32  u32ContentSize;
    MI_BOOL bEndOfFrame;
    MI_U64  u64SeqNum;
} MI_SYS_RawData_t;

typedef struct MI_SYS_MetaData_s
{
    void*  pVirAddr;
    MI_PHY phyAddr;//notice that this is miu bus addr,not cpu bus addr.

    MI_U32 u32Size;
    MI_U32 u32ExtraData;    /*driver special flag*/
    MI_ModuleId_e eDataFromModule;
} MI_SYS_MetaData_t;

typedef struct MI_SYS_FrameIspInfo_s
{
    MI_SYS_FrameIspInfoType_e eType;
    union
    {
        MI_U32 u32GlobalGradient;
    }uIspInfo;
}MI_SYS_FrameIspInfo_t;

//N.B. in MI_SYS_FrameData_t should never support u32Size,
//for other values are enough,and not support u32Size is general standard method.
typedef  struct  MI_SYS_FrameData_s
{
    MI_SYS_FrameTileMode_e eTileMode;
    MI_SYS_PixelFormat_e ePixelFormat;
    MI_SYS_CompressMode_e eCompressMode;
    MI_SYS_FrameScanMode_e eFrameScanMode;
    MI_SYS_FieldType_e eFieldType;
    MI_SYS_FrameData_PhySignalType ePhylayoutType;

    MI_U16 u16Width;
    MI_U16 u16Height;
//in case ePhylayoutType equal to REALTIME_FRAME_DATA, pVirAddr would be MI_SYS_REALTIME_MAGIC_PADDR and phyAddr would be MI_SYS_REALTIME_MAGIC_VADDR

    void* pVirAddr[3];
    MI_PHY phyAddr[3];//notice that this is miu bus addr,not cpu bus addr.
    MI_U32 u32Stride[3];
    MI_U32 u32BufSize;//total size that allocated for this buffer,include consider alignment.

    MI_U16 u16RingBufStartLine;//Valid in case RINGBUF_FRAME_DATA,  u16RingBufStartLine must be LGE than 0 and less than u16Height
    MI_U16 u16RingBufRealTotalHeight;///Valid in case RINGBUF_FRAME_DATA,  u16RingBufStartLine must be LGE than u16Height

    MI_SYS_FrameIspInfo_t stFrameIspInfo;//isp info of each frame
    MI_SYS_WindowRect_t stContentCropWindow;
} MI_SYS_FrameData_t;

typedef struct MI_SYS_FrameDataSubPlane_s
{
    MI_SYS_PixelFormat_e ePixelFormat;
    MI_SYS_CompressMode_e eCompressMode;
    MI_U64 u64FrameId;

    MI_U16 u16Width;
    MI_U16 u16Height;
    void* pVirAddr[2];
    MI_PHY phyAddr[2];
    MI_U16 u16Stride[2];
    MI_U32 u32BufSize;
} MI_SYS_FrameDataSubPlane_t;

typedef struct MI_SYS_FrameDataMultiPlane_s
{
    MI_U8 u8SubPlaneNum;
    MI_SYS_FrameDataSubPlane_t stSubPlanes[MI_SYS_MAX_SUB_PLANE_CNT];
} MI_SYS_FrameDataMultiPlane_t;

typedef  struct  MI_SYS_BufInfo_s
{
    MI_U64 u64Pts;
    MI_U64 u64SidebandMsg;
    MI_SYS_BufDataType_e eBufType;
    MI_BOOL bEndOfStream;
    MI_BOOL bUsrBuf;
    MI_U32 u32SequenceNumber;
    MI_BOOL bDrop;
    union
    {
        MI_SYS_FrameData_t stFrameData;
        MI_SYS_RawData_t stRawData;
        MI_SYS_MetaData_t stMetaData;
        MI_SYS_FrameDataMultiPlane_t stFrameDataMultiPlane;
    };
} MI_SYS_BufInfo_t;

typedef struct MI_SYS_FrameBufExtraConfig_s
{
  //Buf alignment requirement in horizontal
  MI_U16 u16BufHAlignment;
  //Buf alignment requirement in vertical
  MI_U16 u16BufVAlignment;
  //Buf alignment requirement in chroma
  MI_U16 u16BufChromaAlignment;
  //Clear padding flag
  MI_BOOL bClearPadding;
}MI_SYS_FrameBufExtraConfig_t;

typedef struct MI_SYS_BufFrameConfig_s
{
    MI_U16 u16Width;
    MI_U16 u16Height;
    MI_SYS_FrameScanMode_e eFrameScanMode;//
    MI_SYS_PixelFormat_e eFormat;
    MI_SYS_FrameBufExtraConfig_t stFrameBufExtraConf;//set by MI_SYS internal
    //MI_U32 u32Size;//this value will be calculated through others values in this struct
    MI_SYS_CompressMode_e eCompressMode;
}MI_SYS_BufFrameConfig_t;

typedef struct MI_SYS_BufRawConfig_s
{
    MI_U32 u32Size;
}MI_SYS_BufRawConfig_t;

typedef struct MI_SYS_MetaDataConfig_s
{
    MI_U32 u32Size;
}MI_SYS_MetaDataConfig_t;

typedef struct MI_SYS_BufFrameMultiPlaneConfig_s
{
    MI_U8 u8SubPlaneNum;
    MI_SYS_BufFrameConfig_t stFrameCfg;
}MI_SYS_BufFrameMultiPlaneConfig_t;

typedef struct MI_SYS_BufConf_s
{
    MI_SYS_BufDataType_e eBufType;
    MI_U32 u32Flags;   //0 or MI_SYS_MAP_VA
    MI_U64 u64TargetPts;
    union
    {
        MI_SYS_BufFrameConfig_t stFrameCfg;
        MI_SYS_BufRawConfig_t stRawCfg;
        MI_SYS_MetaDataConfig_t stMetaCfg;
        MI_SYS_BufFrameMultiPlaneConfig_t stMultiPlaneCfg;
    };
}MI_SYS_BufConf_t;

typedef struct MI_SYS_ChnInputPortGetBuf_s
{
    MI_SYS_ChnPort_t stChnPort;
    MI_SYS_BufConf_t stBufConf;
    MI_SYS_BufInfo_t stBufInfo;
    MI_SYS_BUF_HANDLE BufHandle;
    MI_S32 s32TimeOutMs;
    MI_U32 u32ExtraFlags;
} MI_SYS_ChnInputPortGetBuf_t;

typedef struct MI_SYS_ChnInputPortPutBuf_s
{
    MI_SYS_BUF_HANDLE BufHandle;
    MI_SYS_BufInfo_t stBufInfo;
    MI_BOOL bDropBuf;
} MI_SYS_ChnInputPortPutBuf_t;

typedef struct MI_SYS_ChnOutputPortGetBuf_s
{
    MI_SYS_ChnPort_t stChnPort;
    MI_SYS_BufInfo_t stBufInfo;
    MI_SYS_BUF_HANDLE BufHandle;
    MI_U32 u32TimeoutMs;
    MI_U32 u32ExtraFlags;
} MI_SYS_ChnOutputPortGetBuf_t;

typedef struct MI_SYS_ChnPortInjectBuf_s
{
    MI_SYS_BUF_HANDLE BufHandle;
    MI_SYS_ChnPort_t stChnPort;
} MI_SYS_ChnPortInjectBuf_t;

typedef struct MI_SYS_Mmap_s
{
    MI_BOOL bCache;
    MI_U32 u32Size;
    void *pVirtualAddress;
    MI_PHY phyAddr;
} MI_SYS_Mmap_t;

typedef struct MI_SYS_FlushInvCache_s
{
    void* pVirtualAddress;
    unsigned long length;
}MI_SYS_FlushInvCache_t;

typedef struct MI_SYS_ChnDupBufHandle_s
{
    MI_SYS_BUF_HANDLE srcBufHandle;
    MI_SYS_BUF_HANDLE dupTargetBufHandle;
} MI_SYS_ChnDupBufHandle_t;

typedef enum
{
    E_MI_SYS_IDR_BUF_TYPE_INPUT_PORT = 0,
    E_MI_SYS_IDR_BUF_TYPE_OUTPUT_PORT,
    E_MI_SYS_IDR_BUF_TYPE_MMAP_TO_USER_SPACE,
    E_MI_SYS_IDR_BUF_TYPE_USER_MMA_ALLOC,
    E_MI_SYS_IDR_BUF_TYPE_USER_PRIVATE_POOL_ALLOC,
} MI_SYS_IDR_BUF_TYPE_e;

struct MI_SYS_BufRef_s;
typedef void *  MI_SYS_DRV_HANDLE;

typedef void (*OnBufRefRelFunc)(struct MI_SYS_BufRef_s *pstBufRef, void *pCBData);

typedef struct MI_SYS_BufRef_s
{
    unsigned int u32MagicNumber;
    struct CamOsListHead_t list;
    struct MI_SYS_BufferAllocation_s *pstBufAllocation;
    OnBufRefRelFunc onRelCB;
    void *pCBData;
    MI_SYS_BufInfo_t bufinfo;
    MI_BOOL is_dumped;
    MI_U32 u32ExtFlag;
    struct CamOsListHead_t list_in_pass;//list head in dev pass ordered by time stampe
    struct CamOsListHead_t list_in_port;//list head in port in FIFO way
    MI_U32 u32ChannelId;
    union
    {
        MI_S64 s64FireTimeStampInNS;// the fire time stamp, MI_SYS_FIRE_IMMEDIATELY for fire immediately
    } uTimeStamp;
    MI_U32 u32InferFrameNumber;
    MI_U32 u32EnqueueInferPulseIndex;//the infer pulse index when buffer eqnueued
    MI_S64 s64BufContentTimeStampInNS; // buffer generated time stamp for performance measurement

    struct MI_SYS_BufRef_s *pstSrcBufRef;//pointer to source buffer reference
} MI_SYS_BufRef_t;

typedef struct MI_SYS_BufHandleIdrData_s
{
    struct CamOsListHead_t list;
    MI_SYS_IDR_BUF_TYPE_e eBufType;
    MI_SYS_BufRef_t *pstBufRef;
    union
    {
        MI_SYS_ChnPort_t stChnPort;
        MI_SYS_DRV_HANDLE miSysDrvHandle;
    };
    MI_U32 u32ExtraFlags;
    MI_S32 s32Pid;
    MI_PHY phyAddr;
    MI_U64 u64BufInfo;
    MI_SYS_BUF_HANDLE BufHandle;
} MI_SYS_BufHandleIdrData_t;

struct vmap_flag {
    void *curr_cache_vaddrbase;
    void *curr_nocache_vaddrbase;
    bool bcache;
};

typedef struct {
    struct {
        union {
            int pid;
            unsigned long cmd;
            unsigned long poll_handle;
        };
        void *curr_cache_vaddrbase;
        void *curr_nocache_vaddrbase;
        long idx;
        long devid;
        union {
            unsigned long mma_base;
            unsigned long log_base;
        };
        unsigned int arg_size;
    };
    char __pad[64];
} linux_ctx __attribute__((aligned(64)));
struct proc_dev {
    int modid;
    int devid;
    int next;
    char cmd_list[];
} __attribute__((aligned(64)));
struct proc_buffer {
    int cost;
    char buf[];
};
#define POLL_FILE_MAX 64
typedef enum {
    E_MI_COMMON_POLL_NOT_READY         = (0x0)     ,
    E_MI_COMMON_FRAME_READY_FOR_READ   = (0x1 << 0),
    E_MI_COMMON_BUFFER_READY_FOR_WRITE = (0x1 << 1),
} MI_COMMON_PollFlag_e;

static int disable_os_adaptor = 0;
module_param(disable_os_adaptor, int, 0644);
MODULE_PARM_DESC(disable_os_adaptor, "Disable Linux-RTOS Adaptor");

/*
 * from ARM Architecture Reference Manual
 *                    ARMv7-A and ARMv7-R edition
 * B3.18.6 Cache maintenance operations, functional group, VMSA
 * Table B3-49 Cache and branch predictor maintenance operations, VMSA
 */
static void flush_cache_area(void *ptr, int len){
    const unsigned long cache_line_size = 64;
    unsigned long iter, end;
    iter = (unsigned long)ptr, end = (unsigned long)ptr + len;
    iter = iter/cache_line_size*cache_line_size;
    end = end/cache_line_size*cache_line_size;
    asm __volatile__("dsb st":::"memory"); /* data sync barrier for store */
    while(iter <= end){
        //asm __volatile__("mcr p15, 0, %0, c7, c11, 1"::"r"(iter):"memory"); /* DCCMVAC: flush to PoU (aka last level cache) */
        asm __volatile__("mcr p15, 0, %0, c7, c10, 1"::"r"(iter):"memory"); /* DCCMVAU: flush to PoC (aka main memory) */
        iter += cache_line_size;
    }
}
static void invalid_cache_area(void *ptr, int len){
    const unsigned long cache_line_size = 64;
    unsigned long iter, end;
    iter = (unsigned long)ptr, end = (unsigned long)ptr + len;
    iter = iter/cache_line_size*cache_line_size;
    end = end/cache_line_size*cache_line_size;
    while(iter <= end){
        asm __volatile__("mcr p15, 0, %0, c7, c6, 1"::"r"(iter):"memory"); /* DCIMVAC: invalidate to PoC */
        iter += cache_line_size;
    }
}
static void flush_and_invalid_cache_area(void *ptr, int len){
    const unsigned long cache_line_size = 64;
    unsigned long iter, end;
    iter = (unsigned long)ptr, end = (unsigned long)ptr + len;
    iter = iter/cache_line_size*cache_line_size;
    end = end/cache_line_size*cache_line_size;
    asm __volatile__("dsb st":::"memory"); /* data sync barrier for store */
    while(iter <= end){
        asm __volatile__("mcr p15, 0, %0, c7, c14, 1"::"r"(iter):"memory"); /* DCCIMVAC: flush & invalid to PoC (aka main memory) */
        iter += cache_line_size;
    }
}

static struct class *device_class;
static struct device *device_list[E_MI_MODULE_ID_MAX + 1];
static struct proc_dev *proc_list[E_MI_MODULE_ID_MAX];
static int proc_node[E_MI_MODULE_ID_MAX]={0};
static int device_major, poll_major;
static struct semaphore device_sem[CTX_NUM];
static struct semaphore ctx_sem;
static spinlock_t ctx_lock;
static DECLARE_BITMAP(ctx_bitmap, 32);
static atomic_t device_ref;
static struct resource *rtk_res;


// IPC_SHARE_ADDR + 0x0000 ~ IPC_SHARE_ADDR + 0x1000  ==> RSQ for basic IPC
// IPC_SHARE_ADDR + 0x1000 ~ IPC_SHARE_ADDR + 0x5000  ==> RSQ for log
// IPC_SHARE_ADDR + 0x5000 ~ IPC_SHARE_ADDR + 0x6000  ==> RSQ for customize settings

static unsigned long mma_base = 0x25500000;
static unsigned long mma_size = 0x02700000;

/*
example:
os_adaptor=mma_base=0x25500000,mma_size=0x02700000
os_adaptor=mma_base=0x21F00000,mma_size=0x01D00000
*/
static bool parse_os_adaptor_config(char *cmdline, unsigned long *mma_base, unsigned long *mma_size)
{
    char *option;

    if(cmdline == NULL)
        goto INVALID_OS_ADAPTOR_CONFIG;

    option = strstr(cmdline, "mma_size=");
    if(option == NULL)
        goto INVALID_OS_ADAPTOR_CONFIG;
    option = strstr(cmdline, "mma_base=");
    if(option == NULL)
        goto INVALID_OS_ADAPTOR_CONFIG;
    sscanf(option, "mma_base=%lx,mma_size=%lx", mma_base, mma_size);

    return true;

INVALID_OS_ADAPTOR_CONFIG:

    return false;
}

int __init setup_os_adaptor(char *cmdline)
{
    if(!parse_os_adaptor_config(cmdline, &mma_base, &mma_size))
    {
        printk(KERN_ERR "error: os_adaptor args invalid\n");
    }
    return 0;
}
early_param("os_adaptor", setup_os_adaptor);

static atomic_t ctx_cost[CTX_NUM][sizeof(unsigned long)*8] = {};
static atomic_t ctx_freq[CTX_NUM] = {};


static struct proc_dir_entry *debug_tools;
static struct proc_dir_entry *proc_root;
struct debug_tool {
    struct proc_dir_entry *entry;
    void *obj;
    ssize_t (*write)(void *obj, const char **args, int count);
    ssize_t (*read)(void *obj);
};
struct debug_tool_freq {
    struct debug_tool dt;
    int interval;
};
struct debug_tool_info {
    struct debug_tool dt;
    const char *version;
};
static ssize_t ctx_cost_erase(void *obj, const char **args, int count){
    atomic_t (*cost)[sizeof(unsigned long)*8] = ctx_cost;
    int i, j;
    for(i = 0; i < CTX_NUM; ++i){
        for(j = 0; j < sizeof(unsigned long)*8; ++j){
            atomic_set(&cost[i][j], 0);
        }
    }
    return 0;
}
static ssize_t ctx_cost_hist(void *obj){
    atomic_t (*cost)[sizeof(unsigned long)*8] = ctx_cost;
    int i, j;
    for(i = 0; i < CTX_NUM; ++i){
        for(j = 0; j < sizeof(unsigned long)*8; ++j){
            printk("CTX_%d|%02d:%d\n", i, j, atomic_read(&cost[i][j]));
        }
        printk("CTX--------------------\n");
    }
    return 0;
}
static ssize_t ctx_cost_freq_setup(void *obj, const char **args, int count){
    struct debug_tool_freq *dtf = obj;
    if(count == 1){
        if (kstrtoint((const char *)args[0], 0, &dtf->interval))
            return -EFAULT;
        printk("freq watch interval=%dms\n", dtf->interval);
        return count;
    }
    return -EINVAL;
}
static ssize_t ctx_cost_freq(void *obj){
    atomic_t *freq = ctx_freq;
    int i;
    for(i = 0; i < CTX_NUM; ++i){
        atomic_xchg(freq+i, 0);
    }
    while(schedule_timeout_interruptible(msecs_to_jiffies(200)) == 0){
        char buf[8*(CTX_NUM+1)] = {0};
        unsigned long rval;
        unsigned long tmp;
        tmp = atomic_xchg(freq, 0);
        rval = sprintf(buf, "%8lu", tmp);
        for(i = 1; i < CTX_NUM; ++i){
            tmp = atomic_xchg(freq+i, 0);
            rval+=sprintf(buf+rval, "|%8lu", tmp);
        }
        printk("CTX_FREQ:%s\n", buf);
    }
    return 0;
}

static ssize_t compile_version_info(void *obj){
    struct debug_tool_info *dti = obj;
    printk("version string:%s\n", dti->version);
    return 0;
}

static struct debug_tool syscall_cost_column = {
    .write = ctx_cost_erase,
    .read = ctx_cost_hist,
};
static struct debug_tool_freq syscall_freq_linear = {
    {
        .obj = &syscall_freq_linear,
        .write = ctx_cost_freq_setup,
        .read = ctx_cost_freq,
    }
};
static struct debug_tool_info info_tool = {
    {
        .obj = &info_tool,
        .read = compile_version_info,
    },
    .version = "version",
};

static unsigned int time_log2(ktime_t start, ktime_t end){
    unsigned int idx = 0;
    unsigned long us = ktime_to_us(ktime_sub(end, start));
    while(us){
        idx = idx + 1;
        us = us >> 1;
    }
    return idx;
}

typedef struct {
    wait_queue_head_t stPollHead;
    unsigned long poll_handle;
    struct list_head list;
} MI_COMMON_PollFileWrapper_t;
static LIST_HEAD(poll_task);

u32 alkaid_notify(u32 arg0, u32 arg1, u32 arg2, u32 arg3){
    int mid = arg1-CTX_BASE;
    if(mid < CTX_NUM){
        up(device_sem+mid);
    }else{
        MI_COMMON_PollFileWrapper_t *f;
        rcu_read_lock();
        list_for_each_entry_rcu(f, &poll_task, list){
            if(f->poll_handle == arg2)
            {
                wake_up(&f->stPollHead);
                break;
            }
        }
        rcu_read_unlock();
    }
    return true;
}
static unsigned long alkaid_poll_wapper(MI_COMMON_PollFileWrapper_t *f, int cmd){
    unsigned long res = 0;
    linux_ctx ctx = {.poll_handle = 0, .idx = -1,};

    ctx.poll_handle = f ? f->poll_handle : 0;
    while(1) {
        if(down_timeout(&ctx_sem, FIVE_SEC) == 0){
            if(CTX_NUM > 1){
                spin_lock(&ctx_lock);
                ctx.idx = find_first_zero_bit(ctx_bitmap, 32);
                if(ctx.idx < CTX_NUM) {
                    set_bit(ctx.idx, ctx_bitmap);
                }
                spin_unlock(&ctx_lock);
            } else {
                ctx.idx = 0;
            }
            if(ctx.idx < CTX_NUM) {
                flush_cache_area(&ctx, sizeof(ctx));
                Chip_Flush_MIU_Pipe();
                break;
            }
        } else {
            printk(KERN_WARNING "dead lock check at %s %d, poll handl:%ld\n", __func__,
            __LINE__, ctx.poll_handle);
        }
    }
    res = CamInterOsSignal(INTEROS_SC_L2R_MI_CALL_START+E_MI_MODULE_ID_MAX, __pa((long)&ctx), -1, cmd);
    while(1) {
        if (down_timeout(device_sem+ctx.idx, FIVE_SEC) == 0) {
            res = CamInterOsSignal(INTEROS_SC_L2R_MI_CALL_START+E_MI_MODULE_ID_MAX, ctx.idx, -1, __MI_DEVICE_QUERY);
            if(res != -2) {
                break;
            } else {
                printk(KERN_ERR "bug found at %s %d\n", __func__, __LINE__);
                *(int*)0 = 0;
            }
        } else {
            printk(KERN_WARNING "dead lock check at %s %d!poll handl:%ld\n", __func__,
            __LINE__, ctx.poll_handle);
        }
    }
    clear_bit(ctx.idx, ctx_bitmap);
    up(&ctx_sem);

    return res;
}
static unsigned long alkaid_poll_init(void){
    return alkaid_poll_wapper(NULL, __MI_DEVICE_POLL_CREATE);
}
static int MI_PollAdp_Open(struct inode *inode, struct file *filp) {
    MI_COMMON_PollFileWrapper_t *fw = kmalloc(sizeof(MI_COMMON_PollFileWrapper_t), GFP_KERNEL);
    if(fw){
        init_waitqueue_head(&fw->stPollHead);
        fw->poll_handle = 0;
        INIT_LIST_HEAD(&fw->list);
        filp->private_data = fw;
        return 0;
    }
    return -ENOMEM;
}

static int MI_PollAdp_Release(struct inode *inode, struct file *filp) {
    MI_COMMON_PollFileWrapper_t *f = filp->private_data;

    list_del_rcu(&f->list);
    synchronize_rcu();
    alkaid_poll_wapper(f, __MI_DEVICE_POLL_RELEASE);
    kfree(f);

    return 0;
}

static long MI_PollAdp_Ioctl(struct file *filp, unsigned int cmd, unsigned long ptr) {
    MI_COMMON_PollFileWrapper_t *f = filp->private_data;
    f->poll_handle = ptr;
    list_add_rcu(&f->list, &poll_task);
    return 0;
}

static unsigned int MI_PollAdp_Poll(struct file *filp, poll_table *wait){
    MI_COMMON_PollFileWrapper_t *f = filp->private_data;
    unsigned int req_events = poll_requested_events(wait);
    unsigned int mask = 0;
    unsigned long ret = 0;

    poll_wait(filp, &f->stPollHead, wait);
    ret = alkaid_poll_wapper(f, __MI_DEVICE_POLL_STATE);
    if(ret & E_MI_COMMON_FRAME_READY_FOR_READ)
        mask |= POLLIN;
    if(ret & E_MI_COMMON_BUFFER_READY_FOR_WRITE)
        mask |= POLLOUT;
    return req_events & mask;
}
static ssize_t proc_read(struct seq_file* q, void* v)
{
    struct proc_dev *pd = q->private;
    unsigned long res = 0;
    unsigned long logpage = get_zeroed_page(GFP_KERNEL);
    linux_ctx ctx = {.log_base = __pa(logpage), .idx = -1, .devid = pd->devid};
    while(1) {
        if(down_timeout(&ctx_sem, FIVE_SEC) == 0){
            if(CTX_NUM > 1){
                spin_lock(&ctx_lock);
                ctx.idx = find_first_zero_bit(ctx_bitmap, 32);
                if(ctx.idx < CTX_NUM) {
                    set_bit(ctx.idx, ctx_bitmap);
                }
                spin_unlock(&ctx_lock);
            } else {
                ctx.idx = 0;
            }
            if(ctx.idx < CTX_NUM) {
                flush_cache_area(&ctx, sizeof(ctx));
                Chip_Flush_MIU_Pipe();
                break;
            }
        } else {
            printk(KERN_WARNING "dead lock check at %s %d!\n", __func__, __LINE__);
        }
    }

    invalid_cache_area((void*)logpage, PAGE_SIZE);

    res = CamInterOsSignal(INTEROS_SC_L2R_MI_CALL_START+pd->modid, __pa((long)&ctx), -1, __MI_DEVICE_PROC_IO);

    do {
        struct proc_buffer *pb = (struct proc_buffer*)logpage;
        down(device_sem+ctx.idx);

        if(pb->cost > 0){
            seq_write(q, pb->buf, pb->cost);
        }else{
            printk(KERN_WARNING "wake up with nothing!\n");
        }
        invalid_cache_area((void*)logpage, PAGE_SIZE);
        res = CamInterOsSignal(INTEROS_SC_L2R_MI_CALL_START+pd->modid, ctx.idx, -1, __MI_DEVICE_PROC_READLOG);
    }while(res);

    clear_bit(ctx.idx, ctx_bitmap);
    up(&ctx_sem);
    free_page(logpage);
    return 0;
}

static ssize_t proc_write(struct file* file, const char __user* user_buf, size_t count, loff_t* ppos)
{
    unsigned long args[32] = {0ul};
    const char *lcmd;
    int c = 0;
    char *p, tc, *kbuf;

    struct seq_file *q = file->private_data;
    struct proc_dev *pd = q->private;

    kbuf = memdup_user(user_buf, count);
    if(!kbuf)
        return -ENOMEM;

    for(p = kbuf, tc = '\0'; tc != '\n' && (c < 32); ++p){
        p += strspn(p, " \t\r\f\v");
        if(*p == '\n')
            break;
        args[c++] = __pa(p);
        p += strcspn(p, " \t\n\r\f\v");
        tc = *p;
        *p = '\0';
    }
    lcmd = __va(args[0]);

    if(c < 32) {
        /* search cmd and exec */
        char *iter = pd->cmd_list, *cmd, *end = pd->cmd_list+pd->next;
        while(iter < end){
            cmd = iter;
            while(*iter++)
                ;
            if(strcmp(cmd, lcmd) == 0){
                unsigned long res = 0;
                unsigned long logpage = get_zeroed_page(GFP_KERNEL);
                linux_ctx ctx = {.cmd = __pa(args), .log_base = __pa(logpage), .arg_size = c, .idx = -1, .devid = pd->devid};
                while(1) {
                    if(down_timeout(&ctx_sem, FIVE_SEC) == 0){
                        if(CTX_NUM > 1){
                            spin_lock(&ctx_lock);
                            ctx.idx = find_first_zero_bit(ctx_bitmap, 32);
                            if(ctx.idx < CTX_NUM) {
                                set_bit(ctx.idx, ctx_bitmap);
                            }
                            spin_unlock(&ctx_lock);
                        } else {
                            ctx.idx = 0;
                        }
                        if(ctx.idx < CTX_NUM) {
                            flush_cache_area(args, sizeof(long)*c);
                            flush_cache_area(kbuf, count);
                            flush_cache_area(&ctx, sizeof(ctx));
                            Chip_Flush_MIU_Pipe();
                            break;
                        }
                    } else {
                        printk(KERN_WARNING "dead lock check at %s %d!\n", __func__, __LINE__);
                    }
                }
                invalid_cache_area((void*)logpage, PAGE_SIZE);
                res = CamInterOsSignal(INTEROS_SC_L2R_MI_CALL_START+pd->modid, __pa((long)&ctx), -1, __MI_DEVICE_PROC_IO);
                do {
                    struct proc_buffer *pb = (struct proc_buffer*)logpage;
                    down(device_sem+ctx.idx);
                    if(pb->cost > 0){
                        printk("%s", pb->buf);
                        memset(pb->buf, 0, pb->cost);
                    }else{
                        printk(KERN_WARNING "wake up with nothing!\n");
                    }
                    invalid_cache_area((void*)logpage, PAGE_SIZE);
                    res = CamInterOsSignal(INTEROS_SC_L2R_MI_CALL_START+pd->modid, ctx.idx, -1, __MI_DEVICE_PROC_READLOG);
                } while(res);

                clear_bit(ctx.idx, ctx_bitmap);
                up(&ctx_sem);
                free_page(logpage);
                kfree(kbuf);
                return count;
            }
        }
    }
    kfree(kbuf);
    return -EINVAL;
}

static int proc_open(struct inode *inode, struct file *file)
{
    return single_open(file, proc_read, PDE_DATA(inode));
}

static const struct file_operations proc_ops = {
    .owner      = THIS_MODULE,
    .open       = proc_open,
    .read       = seq_read,
    .write      = proc_write,
    .llseek     = seq_lseek,
    .release    = single_release,
};

static int fetch_proc(int id, const char *name){
    unsigned long res = 0;
    unsigned long zpage = get_zeroed_page(GFP_KERNEL);
    struct proc_dev *pd = (struct proc_dev*)zpage;
    invalid_cache_area(pd, PAGE_SIZE);
    res = CamInterOsSignal(INTEROS_SC_L2R_MI_CALL_START+id, __pa(pd), -1, __MI_DEVICE_PROC);
    if(res && res != __pa(pd)){
        char path[32] = "mi_";
        struct proc_dir_entry *dir;
        int len = res-__pa(pd);
        proc_list[id] = pd = kmemdup(pd, len, GFP_KERNEL);
        strcat(path, name);
        dir = proc_mkdir(path, proc_root);
        proc_node[id] = 1;
        do{
            //char *iter = pd->cmd_list, *cmd, *end = pd->cmd_list+pd->next;
            sprintf(path, "mi_%s%d", name, pd->devid);
            proc_create_data(path, 0640, dir, &proc_ops, pd);
            /*
            while(iter < end){
                cmd = iter;
                while(*iter++)
                    ;
            }*/
            if((long)(pd->cmd_list+pd->next) != (long)proc_list[id]+len){
                pd = (struct proc_dev*)(pd->cmd_list+pd->next);
            }else{
                break;
            }
        }while(1);
    }else{
        proc_list[id] = NULL;
    }
    free_page(zpage);
    return res;
}

static const char *dev_list[] = {
    "ive",    /* 0 */
    "vdf",    /* 1 */
    "venc",   /* 2 */
    "rgn",    /* 3 */
    "ai",     /* 4 */
    "ao",     /* 5 */
    "vif",    /* 6 */
    "vpe",    /* 7 */
    "vdec",   /* 8 */
    "sys",    /* 9 */
    "fb",     /* 10 */
    "hdmi",   /* 11 */
    "divp",   /* 12 */
    "gfx",    /* 13 */
    "vdisp",  /* 14 */
    "disp",   /* 15 */
    "os",     /* 16 */
    "iae",    /* 17 */
    "md",     /* 18 */
    "od",     /* 19 */
    "shadow", /* 20 */
    "warp",   /* 21 */
    "uac",    /* 22 */
    "ldc",    /* 23 */
    "sd",     /* 24 */
    "panel",  /* 25 */
    "cipher", /* 26 */
    "sensor",    /* 27 */
    "wlan",    /* 28 */
    "ipu",    /* 29 */
    "mipitx",    /* 30 */
    "gyro",    /* 31 */
    "jpd",    /* 32 */
};

static const struct file_operations pfops = {
    .owner      = THIS_MODULE,
    .open       = MI_PollAdp_Open,
    .release    = MI_PollAdp_Release,
    .unlocked_ioctl = MI_PollAdp_Ioctl,
    .poll       = MI_PollAdp_Poll,
    .llseek     = noop_llseek,
};
static int MI_DEVICE_Open(struct inode *inode, struct file *filp) {
    int id = iminor(inode);
    unsigned long res = 0;
    linux_ctx ctx = {.pid = current->pid, .idx = -1,};
    struct vmap_flag * mma_map = NULL;
    mma_map_record_t *pMmaMapRecord;

    while(1) {
        if(down_timeout(&ctx_sem, FIVE_SEC) == 0){
            if(CTX_NUM > 1){
                spin_lock(&ctx_lock);
                ctx.idx = find_first_zero_bit(ctx_bitmap, 32);
                if(ctx.idx < CTX_NUM) {
                    set_bit(ctx.idx, ctx_bitmap);
                }
                spin_unlock(&ctx_lock);
            } else {
                ctx.idx = 0;
            }
            if(ctx.idx < CTX_NUM) {
                flush_cache_area(&ctx, sizeof(ctx));
                Chip_Flush_MIU_Pipe();
                break;
            }
        } else {
            printk(KERN_WARNING "dead lock check at %s %d!\n", __func__, __LINE__);
        }
    }
    res = CamInterOsSignal(INTEROS_SC_L2R_MI_CALL_START+id, __pa((long)&ctx), -1, __MI_DEVICE_CONNECT);
    while(1) {
        if (down_timeout(device_sem+ctx.idx, FIVE_SEC) == 0) {
            res = CamInterOsSignal(INTEROS_SC_L2R_MI_CALL_START+id, ctx.idx, -1, __MI_DEVICE_QUERY);
            if(res != -2) {
                break;
            } else {
                printk(KERN_ERR "bug found at %s %d\n", __func__, __LINE__);
                *(int*)0 = 0;
            }
        } else {
            printk(KERN_WARNING "dead lock check at %s %d!\n", __func__, __LINE__);
        }
    }
    clear_bit(ctx.idx, ctx_bitmap);
    up(&ctx_sem);

    if(proc_node[id] == 0)
    {
        fetch_proc(id, dev_list[id]);
    }

    mma_map = kmalloc(sizeof(struct vmap_flag), GFP_KERNEL);
    if(mma_map == NULL)
    {
        return res;
    }

    CamOsTsemDown(&g_mma_map_hash_semphore);
    CAM_OS_HASH_FOR_EACH_POSSIBLE(g_mma_map_addr_hash, pMmaMapRecord, hentry, current->pid)
    {
        if(pMmaMapRecord->pid == current->pid)
        {
            mma_map->curr_cache_vaddrbase = pMmaMapRecord->mma_cache_vaddr;
            mma_map->curr_nocache_vaddrbase = pMmaMapRecord->mma_nocache_vaddr;
            filp->private_data = mma_map;
            CamOsTsemUp(&g_mma_map_hash_semphore);
            return 0;
        }
    }

    mma_map->bcache = 0;
    filp->private_data = mma_map;
    /* map mma nocache range */
    mma_map->curr_nocache_vaddrbase = (void*)vm_mmap(filp, 0, mma_size, PROT_READ|PROT_WRITE, MAP_SHARED, rtk_res->start);
    if(mma_map->curr_nocache_vaddrbase == NULL)
    {
        return res;
    }

    mma_map->bcache = 1;
    filp->private_data = mma_map;
    /* map mma cache range */
    mma_map->curr_cache_vaddrbase = (void*)vm_mmap(filp, 0, mma_size, PROT_READ|PROT_WRITE, MAP_SHARED, rtk_res->start);
    if(mma_map->curr_cache_vaddrbase == NULL)
    {
        return res;
    }

    pMmaMapRecord = CamOsMemAlloc(sizeof(*pMmaMapRecord));
    if(!pMmaMapRecord)
    {
        CamOsTsemUp(&g_mma_map_hash_semphore);
        return res;
    }

    pMmaMapRecord->mma_cache_vaddr = mma_map->curr_cache_vaddrbase;
    pMmaMapRecord->mma_nocache_vaddr = mma_map->curr_nocache_vaddrbase;
    pMmaMapRecord->pid = current->pid;

    CAM_OS_HASH_ADD(g_mma_map_addr_hash, &pMmaMapRecord->hentry, current->pid);

    CamOsTsemUp(&g_mma_map_hash_semphore);

    return 0;
}

static int MI_DEVICE_Release(struct inode *inode, struct file *filp) {
    int id = iminor(inode);
    struct vmap_flag * mma_map = filp->private_data;
    mma_map_record_t *pMmaMapRecord;
    unsigned long res = 0;
    linux_ctx ctx = {.pid = current->pid, .idx = -1,};
    while(1) {
        if(down_timeout(&ctx_sem, FIVE_SEC) == 0){
            if(CTX_NUM > 1){
                spin_lock(&ctx_lock);
                ctx.idx = find_first_zero_bit(ctx_bitmap, 32);
                if(ctx.idx < CTX_NUM) {
                    set_bit(ctx.idx, ctx_bitmap);
                }
                spin_unlock(&ctx_lock);
            } else {
                ctx.idx = 0;
            }
            if(ctx.idx < CTX_NUM) {
                flush_cache_area(&ctx, sizeof(ctx));
                Chip_Flush_MIU_Pipe();
                break;
            }
        } else {
            printk(KERN_WARNING "dead lock check at %s %d!\n", __func__, __LINE__);
        }
    }
    res = CamInterOsSignal(INTEROS_SC_L2R_MI_CALL_START+id, __pa((long)&ctx), -1, __MI_DEVICE_DISCONNECT);
    while(1) {
        if (down_timeout(device_sem+ctx.idx, FIVE_SEC) == 0) {
            res = CamInterOsSignal(INTEROS_SC_L2R_MI_CALL_START+id, ctx.idx, -1, __MI_DEVICE_QUERY);
            if(res != -2) {
                break;
            } else {
                printk(KERN_ERR "bug found at %s %d\n", __func__, __LINE__);
                *(int*)0 = 0;
            }
        } else {
            printk(KERN_WARNING "dead lock check at %s %d,id=%d\n", __func__, __LINE__,id);
        }
    }
    clear_bit(ctx.idx, ctx_bitmap);
    up(&ctx_sem);
    if(atomic_dec_return(&device_ref) == 0){
        CamInterOsSignalDereg(INTEROS_SC_R2L_MI_NOTIFY, alkaid_notify);
        printk("unregister notify\n");
    }
    if(mma_map)
    {
        kfree(mma_map);
    }
    CamOsTsemDown(&g_mma_map_hash_semphore);
    CAM_OS_HASH_FOR_EACH_POSSIBLE(g_mma_map_addr_hash, pMmaMapRecord, hentry, current->pid)
    {
        if(pMmaMapRecord->pid == current->pid)
        {
             CAM_OS_HASH_DEL(&pMmaMapRecord->hentry);
             CamOsMemRelease(pMmaMapRecord);
        }
    }
    CamOsTsemUp(&g_mma_map_hash_semphore);
    return res;
}

static unsigned long vir2phy(struct task_struct *curr, void *ptr){
    unsigned long addr = (unsigned long)ptr;
    pgd_t *pgd = pgd_offset(curr->mm,addr);
    pud_t *pud = pud_offset(pgd,addr);
    pmd_t *pmd = pmd_offset(pud,addr);
    pte_t *pte = pmd_page_vaddr(*pmd)+pte_index(addr);
    return __pfn_to_phys(pte_pfn(*pte))+(addr&~PAGE_MASK);
}

static void MI_MMA_Flush(struct vmap_flag * mma_map, MI_SYS_BufInfo_t stBufInfo)
{
    void * pVirtualAddress = NULL;
    int u32BufSize = 0;
    if(E_MI_SYS_BUFDATA_RAW == stBufInfo.eBufType)
    {
        pVirtualAddress = stBufInfo.stRawData.pVirAddr;
        u32BufSize = stBufInfo.stRawData.u32BufSize;
    }
    else if(E_MI_SYS_BUFDATA_FRAME == stBufInfo.eBufType)
    {
        pVirtualAddress = stBufInfo.stFrameData.pVirAddr[0];
        u32BufSize = stBufInfo.stFrameData.u32BufSize;
    }
    else if(E_MI_SYS_BUFDATA_META == stBufInfo.eBufType)
    {
        pVirtualAddress = stBufInfo.stMetaData.pVirAddr;
        u32BufSize = stBufInfo.stMetaData.u32Size;
    }
    else if(E_MI_SYS_BUFDATA_MULTIPLANE == stBufInfo.eBufType)
    {
        int i = 0;
        for(i = 0 ; i < stBufInfo.stFrameDataMultiPlane.u8SubPlaneNum ; i ++)
        {
            pVirtualAddress = stBufInfo.stFrameDataMultiPlane.stSubPlanes[i].pVirAddr[0];
            u32BufSize = stBufInfo.stFrameDataMultiPlane.stSubPlanes[i].u32BufSize;
            if((pVirtualAddress >= mma_map->curr_cache_vaddrbase && pVirtualAddress < mma_map->curr_cache_vaddrbase + mma_base) || (pVirtualAddress >= mma_map->curr_nocache_vaddrbase && pVirtualAddress < mma_map->curr_nocache_vaddrbase + mma_base))
            {
                CamOsMemFlush(pVirtualAddress,u32BufSize);
            }
        }
        return;
    }
    if((pVirtualAddress >= mma_map->curr_cache_vaddrbase && pVirtualAddress < mma_map->curr_cache_vaddrbase + mma_base) || (pVirtualAddress >= mma_map->curr_nocache_vaddrbase && pVirtualAddress < mma_map->curr_nocache_vaddrbase + mma_base))
    {
        CamOsMemFlush(pVirtualAddress,u32BufSize);
    }
    return;
}

static void MI_MMA_Invalidate(struct vmap_flag * mma_map, MI_SYS_BufInfo_t stBufInfo)
{
    void * pVirtualAddress = NULL;
    int u32BufSize = 0;
    if(E_MI_SYS_BUFDATA_RAW == stBufInfo.eBufType)
    {
        pVirtualAddress = stBufInfo.stRawData.pVirAddr;
        u32BufSize = stBufInfo.stRawData.u32BufSize;
    }
    else if(E_MI_SYS_BUFDATA_FRAME == stBufInfo.eBufType)
    {
        pVirtualAddress = stBufInfo.stFrameData.pVirAddr[0];
        u32BufSize = stBufInfo.stFrameData.u32BufSize;
    }
    else if(E_MI_SYS_BUFDATA_META == stBufInfo.eBufType)
    {
        pVirtualAddress = stBufInfo.stMetaData.pVirAddr;
        u32BufSize = stBufInfo.stMetaData.u32Size;
    }
    else if(E_MI_SYS_BUFDATA_MULTIPLANE == stBufInfo.eBufType)
    {
        int i = 0;
        for(i = 0 ; i < stBufInfo.stFrameDataMultiPlane.u8SubPlaneNum ; i ++)
        {
            pVirtualAddress = stBufInfo.stFrameDataMultiPlane.stSubPlanes[i].pVirAddr[0];
            u32BufSize = stBufInfo.stFrameDataMultiPlane.stSubPlanes[i].u32BufSize;
            if((pVirtualAddress >= mma_map->curr_cache_vaddrbase && pVirtualAddress < mma_map->curr_cache_vaddrbase + mma_base) || (pVirtualAddress >= mma_map->curr_nocache_vaddrbase && pVirtualAddress < mma_map->curr_nocache_vaddrbase + mma_base))
            {
                invalid_cache_area(pVirtualAddress,u32BufSize);
            }
        }
        return;
    }
    if((pVirtualAddress >= mma_map->curr_cache_vaddrbase && pVirtualAddress < mma_map->curr_cache_vaddrbase + mma_base) || (pVirtualAddress >= mma_map->curr_nocache_vaddrbase && pVirtualAddress < mma_map->curr_nocache_vaddrbase + mma_base))
    {
        invalid_cache_area(pVirtualAddress,u32BufSize);
    }
    return;
}

static long MI_DEVICE_Ioctl(struct file *filp, unsigned int cmd, unsigned long ptr) {
    int id = iminor(file_inode(filp));
    long rval = -EIO;
    struct vmap_flag * mma_map = filp->private_data;
    if(_IOC_TYPE(cmd) == 'i') {
        unsigned long res = 0;
        atomic_t (*cost)[sizeof(unsigned long)*8];
        atomic_t *freq;
        ktime_t t1, t2;
        t1 = ktime_get();
        if(ptr) {
            linux_ctx ctx = {.curr_cache_vaddrbase = mma_map->curr_cache_vaddrbase,.curr_nocache_vaddrbase = mma_map->curr_nocache_vaddrbase, .idx = -1, .mma_base = mma_base};
            struct {
                int len;
                unsigned long long ptr;
            } tr;
            void *arg = NULL;
            if (copy_from_user((char *)&tr, (void*)ptr, sizeof(tr)))
                return -EFAULT;
            if(tr.len > _IOC_SIZE(cmd)) {
                printk(KERN_ERR "write cmd(0x%08x) overflow!", cmd);
                return -EINVAL;
            }

            if(tr.len > 4096) {
                printk(KERN_WARNING "write cmd(0x%08x) Send Big Data size(%d)!", cmd, tr.len);
            }

            if(_IOC_DIR(cmd) & _IOC_WRITE) {
                if(tr.len == 0) {
                    printk(KERN_ERR "write cmd(0x%08x) send null data!", cmd);
                    return -EINVAL;
                }
                arg = memdup_user((void*)(long)tr.ptr, tr.len);
                if(!arg)
                    return -ENOMEM;
                if(_IOC_DIR(cmd) & _IOC_READ) {
                    flush_and_invalid_cache_area(arg, tr.len);
                }else{
                    flush_cache_area(arg, tr.len);
                }
            } else if(_IOC_DIR(cmd) & _IOC_READ) {
                arg = kmalloc(tr.len+sizeof(long), GFP_KERNEL);
                if(!arg)
                    return -ENOMEM;
                invalid_cache_area(arg, tr.len);
            } else {
                printk(KERN_ERR "send a buffer to cmd(0x%08x) with_IOC_TYPE_NONE!\n", cmd);
                return -EINVAL;
            }
            ctx.arg_size = _IOC_SIZE(cmd);
            while(1) {
                if(down_timeout(&ctx_sem, FIVE_SEC) == 0){
                    if(CTX_NUM > 1){
                        spin_lock(&ctx_lock);
                        ctx.idx = find_first_zero_bit(ctx_bitmap, 32);
                        if(ctx.idx < CTX_NUM) {
                            set_bit(ctx.idx, ctx_bitmap);
                        }
                        spin_unlock(&ctx_lock);
                    } else {
                        ctx.idx = 0;
                    }
                    if(ctx.idx < CTX_NUM) {
                        flush_cache_area(&ctx, sizeof(ctx));
                        break;
                    }
                } else {
                    printk(KERN_WARNING "dead lock check at %s %d!dev:%d, cmd:%d\n", __func__,
                        __LINE__, id, _IOC_NR(cmd));
                }
            }
            Chip_Flush_MIU_Pipe();
            if(id == E_MI_MODULE_ID_SYS)
            {
                switch(_IOC_NR(cmd))
                {
                    case E_MI_SYS_FLUSH_INV_CACHE:
                        {
                            void * pVirtualAddress = ((MI_SYS_FlushInvCache_t *)(unsigned long)arg)->pVirtualAddress;
                            int u32Size = ((MI_SYS_FlushInvCache_t *)(unsigned long)arg)->length;
                            u32Size = ALIGN_UP(u32Size, PAGE_SIZE);
                            if((pVirtualAddress >= mma_map->curr_cache_vaddrbase && pVirtualAddress < mma_map->curr_cache_vaddrbase + mma_base) || (pVirtualAddress >= mma_map->curr_nocache_vaddrbase && pVirtualAddress < mma_map->curr_nocache_vaddrbase + mma_base))
                            {
                                CamOsMemFlush(pVirtualAddress, u32Size);
                            }
                        }
                        break;
                    case E_MI_SYS_CMD_CHN_INPUT_PORT_PUT_BUF:
                        {
                            MI_SYS_ChnInputPortPutBuf_t * pstChnPortPutBuf = (MI_SYS_ChnInputPortPutBuf_t *)(unsigned long)arg;
                            MI_MMA_Flush(mma_map,pstChnPortPutBuf->stBufInfo);
                        }
                        break;
                    default:
                        break;
                }
            }
            while(1) {
                res = CamInterOsSignal(INTEROS_SC_L2R_MI_CALL_START+id, __pa((long)&ctx), cmd, __pa((unsigned long)arg));
                if (down_timeout(device_sem+ctx.idx, FIVE_SEC) == 0) {
                    res = CamInterOsSignal(INTEROS_SC_L2R_MI_CALL_START+id, __pa((long)&ctx), cmd, __pa((unsigned long)arg));
                    if(res != -2) {
                        break;
                    } else {
                        printk(KERN_ERR "bug found at %s %d\n", __func__, __LINE__);
                        *(int*)0 = 0;
                    }
                } else {
                    printk(KERN_WARNING "dead lock check at %s %d!dev:%d, cmd:%d\n", __func__,
                        __LINE__, id, _IOC_NR(cmd));
                }
            }
            if(id == E_MI_MODULE_ID_SYS && 0 == res)
            {
                switch(_IOC_NR(cmd))
                {
                    case E_MI_SYS_CMD_CHN_INPUT_PORT_GET_BUF:
                        {
                            MI_SYS_ChnInputPortGetBuf_t * pstChnPortGetBuf = (MI_SYS_ChnInputPortGetBuf_t *)(unsigned long)arg;
                            MI_MMA_Invalidate(mma_map,pstChnPortGetBuf->stBufInfo);
                        }
                        break;
                    case E_MI_SYS_CMD_CHN_OUTPUT_PORT_GET_BUF:
                        {
                           MI_SYS_ChnOutputPortGetBuf_t * pstChnPortGetBuf = (MI_SYS_ChnOutputPortGetBuf_t *)(unsigned long)arg;
                           MI_MMA_Invalidate(mma_map,pstChnPortGetBuf->stBufInfo);
                        }
                        break;
                    case E_MI_SYS_CMD_MMAP:
                        {
                             void * pVirtualAddress = ((MI_SYS_Mmap_t *)(unsigned long)arg)->pVirtualAddress;
                             int u32Size = ((MI_SYS_Mmap_t *)(unsigned long)arg)->u32Size;
                             u32Size = ALIGN_UP(u32Size, PAGE_SIZE);
                             if((pVirtualAddress >= mma_map->curr_cache_vaddrbase && pVirtualAddress < mma_map->curr_cache_vaddrbase + mma_base) || (pVirtualAddress >= mma_map->curr_nocache_vaddrbase && pVirtualAddress < mma_map->curr_nocache_vaddrbase + mma_base))
                             {
                                 invalid_cache_area(pVirtualAddress, u32Size);
                             }
                        }
                        break;
                    default:
                        break;
                }
            }
            clear_bit(ctx.idx, ctx_bitmap);
            up(&ctx_sem);
            cost = ctx_cost+ctx.idx;
            freq = ctx_freq+ctx.idx;
            rval = res;
            if(_IOC_DIR(cmd) & _IOC_READ) {
                invalid_cache_area(arg, tr.len);
                if (copy_to_user((char*)(long)tr.ptr, arg, tr.len))
                    return -EFAULT;
            }
            kfree(arg);
        } else {
            int ctxid;
            while(1) {
                if(down_timeout(&ctx_sem, FIVE_SEC) == 0){
                    if(CTX_NUM > 1){
                        spin_lock(&ctx_lock);
                        ctxid = find_first_zero_bit(ctx_bitmap, 32);
                        if(ctxid < CTX_NUM) {
                            set_bit(ctxid, ctx_bitmap);
                        }
                        spin_unlock(&ctx_lock);
                        if(ctxid < CTX_NUM)
                            break;
                    } else {
                        ctxid = 0;
                        break;
                    }
                } else {
                    printk(KERN_WARNING "dead lock check at %s %d!dev:%d, cmd:%d\n", __func__,
                        __LINE__, id, _IOC_NR(cmd));
                }
            }
            while(1) {
                res = CamInterOsSignal(INTEROS_SC_L2R_MI_CALL_START+id, ctxid, cmd, 0);
                if (down_timeout(device_sem+ctxid, FIVE_SEC) == 0) {
                    res = CamInterOsSignal(INTEROS_SC_L2R_MI_CALL_START+id, ctxid, cmd, 0);
                    if(res != -2) {
                        break;
                    } else {
                        printk(KERN_ERR "bug found at %s %d\n", __func__, __LINE__);
                        *(int*)0 = 0;
                    }
                } else {
                    printk(KERN_WARNING "dead lock check at %s %d!dev:%d, cmd:%d\n", __func__,
                        __LINE__, id, _IOC_NR(cmd));
                }
            }
            clear_bit(ctxid, ctx_bitmap);
            up(&ctx_sem);
            cost = ctx_cost+ctxid;
            freq = ctx_freq+ctxid;
            rval = res;
        }
        t2 = ktime_get();
        atomic_inc(freq);
        atomic_inc(*cost+time_log2(t1, t2));
    }else{
        unsigned long *vir = mma_map->curr_cache_vaddrbase+cmd;
        unsigned long uval;
        get_user(uval, vir);
        printk("uva:%p,phy:%lx,off=%x,uval=%lx\n", vir, vir2phy(current, vir), cmd, uval);
        rval = 0;
    }
    if(proc_node[id] == 0)
    {
        fetch_proc(id, dev_list[id]);
    }

    return rval;
}
static int MI_DEVICE_Mmap(struct file *file, struct vm_area_struct *vma) {
    static const struct vm_operations_struct vma_ops = {};
    size_t size = vma->vm_end - vma->vm_start;
    struct vmap_flag * mma_map = file->private_data;

    if(mma_map->bcache)
    {
        vma->vm_page_prot = vm_get_page_prot(vma->vm_flags);
    }
    else
    {
        vma->vm_page_prot = pgprot_writecombine(vma->vm_page_prot);
    }
    vma->vm_ops = &vma_ops;

    /* Remap-pfn-range will mark the range VM_IO */
    if (remap_pfn_range(vma,
                        vma->vm_start,
                        vma->vm_pgoff,
                        size,
                        vma->vm_page_prot)) {
        return -EAGAIN;
    }
    return 0;
}
static const struct file_operations fops = {
    .owner      = THIS_MODULE,
    .open       = MI_DEVICE_Open,
    .release    = MI_DEVICE_Release,
    .unlocked_ioctl = MI_DEVICE_Ioctl,
    .mmap       = MI_DEVICE_Mmap,
    .llseek     = noop_llseek,
};
module_param(mma_base,ulong,0644);
module_param(mma_size,ulong,0644);
static unsigned long rtk_base(void){
    return CamInterOsSignal(INTEROS_SC_L2R_HANDSHAKE, 0, 0, 0);
}



static ssize_t debug_tool_read(struct seq_file* q, void* v)
{
    struct debug_tool *dt = q->private;
    if(dt->read)
        return dt->read(dt->obj);
    return -EIO;
}

static ssize_t debug_tool_write(struct file* file, const char __user* user_buf, size_t count, loff_t* ppos)
{
    const char *args[32] = {NULL};
    int c = 0;
    char *p, tc, *kbuf;

    struct seq_file *q = file->private_data;
    struct debug_tool *dt = q->private;

    if(!dt->write)
        return -EIO;

    kbuf = memdup_user(user_buf, count);
    if(!kbuf)
        return -ENOMEM;

    for(p = kbuf, tc = '\0'; tc != '\n' && (c < 32); ++p){
        p += strspn(p, " \t\r\f\v");
        if(*p == '\n')
            break;
        args[c++] = p;
        p += strcspn(p, " \t\n\r\f\v");
        tc = *p;
        *p = '\0';
    }

    if(c < 32) {
        dt->write(dt->obj,args,c);
        kfree(kbuf);
        return count;
    }
    kfree(kbuf);
    return -EINVAL;
}

static int debug_tool_open(struct inode *inode, struct file *file)
{
    return single_open(file, debug_tool_read, PDE_DATA(inode));
}

static const struct file_operations debug_tool_ops = {
    .owner      = THIS_MODULE,
    .open       = debug_tool_open,
    .read       = seq_read,
    .write      = debug_tool_write,
    .llseek     = seq_lseek,
    .release    = single_release,
};

static bool debug_tool_create(const char *name, struct debug_tool *dt){
    dt->entry = proc_create_data(name, 0640, debug_tools, &debug_tool_ops, dt);
    if (!dt->entry)
    {
        printk(KERN_ERR "failed  to  create  procfs  file  %s.\n",name);
        return false;
    }
    return true;
}

static void debug_tool_delete(struct debug_tool *dt){
    proc_remove(dt->entry);
}

static int __init linux_adaptor_init(void) {
    int err = 0, i;

    if (disable_os_adaptor)
        return 0;

    mma_base = CamInterOsSignal(INTEROS_SC_L2R_MI_MMA_BASE, 0, 0, 0);
    mma_size = CamInterOsSignal(INTEROS_SC_L2R_MI_MMA_SIZE, 0, 0, 0);

    printk(KERN_WARNING "alkaid_get_mma_info: mma_base = 0x%lx  mma_size = 0x%lx\n",mma_base,mma_size);

    if (!mma_base || !mma_size)
    {
        return -EINVAL;
    }

    if(E_MI_MODULE_ID_MAX < sizeof(dev_list)/sizeof(*dev_list)){
        return -EINVAL;
    }
    err = -EIO;
    device_major = register_chrdev(0, "rtos-adaptor", &fops);
    if(device_major <= 0)
        goto fail_register_chrdev;

    device_class = class_create(THIS_MODULE, "rtos-adaptor");
    err = PTR_ERR(device_class);
    if (IS_ERR(device_class))
        goto fail_class_create;

    proc_root = proc_mkdir("mi_modules", NULL);
    for(i = 0; i < E_MI_MODULE_ID_MAX; ++i){
        device_list[i] = device_create(device_class,
                                    NULL,
                                    MKDEV(device_major, i),
                                    device_list+i,
                                    "mi_%s",
                                    dev_list[i]);
        err = fetch_proc(i, dev_list[i]);
        if(err == 0)
            goto fail_register_chrdev2;
    }
    poll_major = register_chrdev(0, "poll-dev", &pfops);
    if(poll_major <= 0)
        goto fail_register_chrdev2;

    device_list[E_MI_MODULE_ID_MAX] = device_create(device_class,
                                    NULL,
                                    MKDEV(poll_major, 0),
                                    device_list+E_MI_MODULE_ID_MAX,
                                    "mi_poll");

    if(device_list[E_MI_MODULE_ID_MAX] == NULL)
        goto fail_create_poll_dev;

    CamOsTsemInit(&g_mma_map_hash_semphore,1);

    for(i = 0; i < CTX_NUM; ++i)
        sema_init(device_sem+i, 0);
    sema_init(&ctx_sem, CTX_NUM);
    atomic_set(&device_ref, 0);
    spin_lock_init(&ctx_lock);
    for(i = 0; i < 32; ++i)
        clear_bit(i, ctx_bitmap);
    rtk_res = request_mem_region(mma_base, mma_size, "mma");
    if(!rtk_res){
        goto fail_mem_req;
    }
    printk("#map req:(0x%x,0x%x)|0x%lx\n", rtk_res->start, resource_size(rtk_res), rtk_base());
    debug_tools = proc_mkdir("adaptor-debug-tools", NULL);
    if(!debug_tools)
        goto fail_create_debug_tools;

    if(!debug_tool_create("syscall_cost", &syscall_cost_column))
        goto fail_create_syscall_cost;

    if(!debug_tool_create("syscall_freq", &syscall_freq_linear.dt))
        goto fail_create_syscall_freq;

    if(!debug_tool_create("info_tool", &info_tool.dt))
        goto fail_create_info_tool;

    CamInterOsSignalReg(INTEROS_SC_R2L_MI_NOTIFY, alkaid_notify, "alkaid_notify");

    printk("register notify\n");

    alkaid_poll_init();
    printk("poll init dev\n");

    printk("linux-adaptor init success!(%s)\n", __TIME__);
    return 0;

fail_create_info_tool:
    printk(KERN_ERR "create info tool failed!\n");
    debug_tool_delete(&syscall_freq_linear.dt);
fail_create_syscall_freq:
    printk(KERN_ERR "create syscall freq analyzer failed!\n");
    debug_tool_delete(&syscall_cost_column);
fail_create_syscall_cost:
    printk(KERN_ERR "create syscall cost analyzer failed!\n");
    proc_remove(debug_tools);
fail_create_debug_tools:
    printk(KERN_ERR "proc mkdir failed!\n");
    release_mem_region(rtk_res->start, mma_size);
fail_mem_req:
    printk(KERN_ERR "request mem failed\n");
    device_destroy(device_class, MKDEV(poll_major, 0));
fail_create_poll_dev:
    printk(KERN_ERR "create poll dev failed\n");
    unregister_chrdev(poll_major, "poll-dev");
fail_register_chrdev2:
    printk(KERN_ERR "unable to get mi device\n");
    for(i = 0; i < E_MI_MODULE_ID_MAX; ++i){
        if(device_list[i])
            device_destroy(device_class, MKDEV(device_major, i));
    }
    unregister_chrdev(device_major, "rtos-adaptor");
fail_class_create:
    printk(KERN_ERR "fail create class\n");
fail_register_chrdev:
    printk(KERN_ERR "unable to get mi device\n");
    class_destroy(device_class);
    return err;
}
#ifdef CONFIG_DEFERRED_LINUX_ADAPTOR_INIT
deferred_module_init(linux_adaptor_init)
//module_init(linux_adaptor_init)
#else
module_init(linux_adaptor_init)
#endif

static void __exit linux_adaptor_exit(void) {
    int i;
    debug_tool_delete(&info_tool.dt);
    debug_tool_delete(&syscall_freq_linear.dt);
    debug_tool_delete(&syscall_cost_column);
    proc_remove(debug_tools);
    release_mem_region(rtk_res->start, mma_size);
    device_destroy(device_class, MKDEV(poll_major, 0));
    unregister_chrdev(poll_major, "poll-dev");
    for(i = 0; i < E_MI_MODULE_ID_MAX; ++i){
        if(device_list[i])
            device_destroy(device_class, MKDEV(device_major, i));
    }
    unregister_chrdev(device_major, "rtos-adaptor");
    class_destroy(device_class);
    CamOsTsemDeinit(&g_mma_map_hash_semphore);
}
module_exit(linux_adaptor_exit)

MODULE_LICENSE("GPL v2");
